<?php

namespace App\Http\Controllers;

use App\Models\Product;
use Illuminate\Http\Request;
use Faker\Factory as Faker;

class FakeDataController extends Controller
{
    public function generateFakeProducts(Request $request)
    {
        $request->validate([
            'count' => 'sometimes|integer|min:1|max:100',
            'brand' => 'sometimes|string|in:nokia,samsung,apple,xiaomi,oppo'
        ]);

        $count = $request->count ?? 10;
        $brand = $request->brand ?? 'nokia';

        $products = $this->generateProducts($count, $brand);

        return response()->json([
            'success' => true,
            'message' => "Generated {$count} {$brand} products",
            'products' => $products
        ]);
    }

    private function generateProducts($count, $brand)
    {
        $faker = Faker::create();
        $generated = [];

        for ($i = 0; $i < $count; $i++) {
            $productData = $this->generateProductData($brand, $faker);
            $product = Product::create($productData);
            $generated[] = $product;
        }

        return $generated;
    }

    private function generateProductData($brand, $faker)
    {
        // Similar to seeder logic but for API use
        $brands = [
            'nokia' => ['models' => ['105', '106', '110', '3310', '5310', '6300', '2720', '8000'], 'prefix' => 'NOK'],
            'samsung' => ['models' => ['Galaxy A10', 'Galaxy A20', 'Galaxy M10', 'Galaxy S10'], 'prefix' => 'SAM'],
            'apple' => ['models' => ['iPhone 11', 'iPhone 12', 'iPhone 13', 'iPhone SE'], 'prefix' => 'APL'],
            'xiaomi' => ['models' => ['Redmi 9', 'Redmi 10', 'Redmi Note 10', 'Poco M3'], 'prefix' => 'XIA'],
            'oppo' => ['models' => ['A5', 'F19', 'Reno 6'], 'prefix' => 'OPP'],
        ];

        $brandInfo = $brands[$brand] ?? $brands['nokia'];
        $model = $faker->randomElement($brandInfo['models']);

        return [
            'serial_no' => $brandInfo['prefix'] . '-' . date('Y') . '-' . str_pad(mt_rand(1, 999999), 6, '0', STR_PAD_LEFT),
            'imei' => '35' . str_pad(mt_rand(1, 9999999999999), 13, '0', STR_PAD_LEFT),
            'product_name' => ucfirst($brand) . ' ' . $model,
            'condition' => 'New',
            'status' => 'In Warehouse',
            'quality' => $faker->randomElement(['Authentic', 'Premium', 'Standard']),
            'description' => "Brand new " . ucfirst($brand) . " {$model} with warranty",
            'distributor_name' => $faker->company,
            'distributor_code' => 'D-API-' . $faker->randomNumber(4),
            'color' => $faker->randomElement(['Black', 'White', 'Blue', 'Red', 'Silver']),
            'scanned_by' => 'API-Generator',
            'scanned_at' => now(),
        ];
    }
}