<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Wishlist;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class WishlistController extends Controller
{
    /**
     * Toggle a product in the user's wishlist.
     *
     * @param  \App\Models\Product  $product
     * @return \Illuminate\Http\JsonResponse
     */
    public function toggle(Product $product)
    {
        if (Auth::check()) {
            // Authenticated user logic
            $user = Auth::user();
            $wishlistItem = $user->wishlist()->where('product_id', $product->id)->first();

            if ($wishlistItem) {
                $wishlistItem->delete();
                $status = 'removed';
                $message = 'Product removed from your wishlist.';
            } else {
                $user->wishlist()->create(['product_id' => $product->id]);
                $status = 'added';
                $message = 'Product added to your wishlist!';
            }
            $wishlistCount = $user->wishlist()->count();

        } else {
            // Guest user logic (using session)
            $wishlist = session()->get('wishlist', []);
            if (isset($wishlist[$product->id])) {
                unset($wishlist[$product->id]);
                $status = 'removed';
                $message = 'Product removed from your wishlist.';
            } else {
                $wishlist[$product->id] = $product->id;
                $status = 'added';
                $message = 'Product added to your wishlist!';
            }
            session()->put('wishlist', $wishlist);
            $wishlistCount = count($wishlist);
        }

        return response()->json([
            'success' => true,
            'status' => $status,
            'message' => $message,
            'wishlistCount' => $wishlistCount
        ]);
    }

    // You can keep other methods like index, count etc. here if needed.
    public function index()
    {
        $wishlistItems = collect();

        if (Auth::check()) {
            // For authenticated users, get from database
            $user = Auth::user();
            $wishlistItems = $user->wishlist()->with('product')->latest()->get();
        } else {
            // For guest users, get from session
            $wishlistProductIds = session()->get('wishlist', []);
            if (!empty($wishlistProductIds)) {
                $products = Product::whereIn('id', $wishlistProductIds)->get();
                // We need to wrap products in a structure similar to Wishlist model
                // to maintain consistency in the blade view.
                $wishlistItems = $products->map(function ($product) {
                    $item = new Wishlist();
                    $item->setRelation('product', $product);
                    return $item;
                });
            }
        }

        return view('frontend.pages.wishlist', compact('wishlistItems'));
    }
}
