<?php

namespace App\Http\Controllers;


use App\Models\Category;
use App\Models\Scan;
use Illuminate\Support\Str;
use App\Models\Product;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;

class ProductController extends Controller
{
    /**
     * Display a list of products for the public frontend.
     */
    public function index(Request $request)
    {
        $query = Product::query();

        // Handle search
        if ($request->filled('search')) {
            $search = $request->input('search');
            $query->where('product_name', 'like', "%{$search}%");
        }

        // Handle category filtering
        if ($request->filled('category')) {
            $categorySlugs = $request->input('category');
            if (!is_array($categorySlugs)) {
                $categorySlugs = [$categorySlugs];
            }

            $query->whereHas('category', function ($q) use ($categorySlugs) {
                $q->whereIn('slug', $categorySlugs)
                  ->orWhereHas('parent', fn($q) => $q->whereIn('slug', $categorySlugs));
            });
        }

        // Handle brand filtering
        if ($request->filled('brands')) {
            $brands = $request->input('brands', []);
            if (!is_array($brands)) {
                $brands = [$brands];
            }
            $query->where(function ($q) use ($brands) {
                collect($brands)->each(fn($brand) => $q->orWhere('product_name', 'like', $brand . '%'));
            });
        }

        // Handle rating filtering
        if ($request->filled('ratings')) {
            // The 'rating' column does not exist in the 'products' table yet.
            // This filter is commented out to prevent errors.
            // Example:
            // $ratings = is_array($request->input('ratings')) ? $request->input('ratings') : [$request->input('ratings')];
            // $query->where(fn($q) => collect($ratings)->each(fn($rating) => $q->orWhere('rating', '>=', $rating)));

            // To enable it, add a 'rating' column to your products table migration and seeder.
        }

        // Handle price range filtering
        if ($request->filled('price_min')) {
            $query->whereRaw('COALESCE(NULLIF(discount_price, 0), price) >= ?', [$request->input('price_min')]);
        }
        if ($request->filled('price_max')) {
            // Ensure we include products with the max price
            $maxPrice = $request->input('price_max');
            $query->whereRaw('COALESCE(NULLIF(discount_price, 0), price) <= ?', [$maxPrice]);
        }

        // Handle sorting
        $sort = $request->input('sort', 'latest');
        switch ($sort) {
            case 'price-low':
                $query->orderByRaw('COALESCE(NULLIF(discount_price, 0), price) asc');
                break;
            case 'price-high':
                $query->orderByRaw('COALESCE(NULLIF(discount_price, 0), price) desc');
                break;
            case 'name':
                $query->orderBy('product_name', 'asc');
                break;
            default:
                $query->latest();
        }

        $products = $query->with('category')->paginate(9)->withQueryString();
        // Determine the true maximum price by considering both regular and discount prices.
        $maxPrice = Product::query()
            ->selectRaw('MAX(IF(discount_price IS NOT NULL AND discount_price > 0, discount_price, price)) as max_price')
            ->value('max_price');
        $allCategories = Category::withCount('products')->whereNull('parent_id')->get();
        $brands = Product::selectRaw("SUBSTRING_INDEX(product_name, ' ', 1) as name")->distinct()->get();

        // For AJAX requests, we return the rendered product list and pagination
        if ($request->ajax()) {
            return response()->json([
                'products_html' => view('frontend.pages.partials.products-grid', compact('products'))->render(),
                'pagination_html' => $products->links()->toHtml(),
            ]);
        }

        return view('frontend.pages.products', compact('products', 'maxPrice', 'allCategories', 'brands'));
    }

    /**
     * Display a single product for the public frontend.
     */
    public function show(Request $request, Product $product)
    {
        // Eager load category to prevent extra queries
        $product->load('category');

        // New logic: The product itself contains all color variants in its gallery.
        // We create a "product group" from this single product's data.
        $galleryImages = $product->gallery_images ?? [];
        $productColors = is_array($product->color) ? $product->color : [$product->color];

        // If gallery is empty, create a default from the product's main image and color
        if (empty($galleryImages) && $product->image) {
            // Use the first color from the product's color array, or a default.
            $defaultColor = !empty($productColors) ? $productColors[0] : 'Default';
            if ($defaultColor) {
                $galleryImages[] = ['color' => $defaultColor, 'image' => $product->image];
            }
        }

        // Determine the initial selected color from the request, or default to the first one.
        $requestedColor = $request->query('color'); // e.g., "Black" or "black"
        $initialColorItem = null;

        if ($requestedColor) {
            // Find the requested color case-insensitively
            $initialColorItem = collect($galleryImages)->first(function ($item) use ($requestedColor) {
                return strtolower($item['color'] ?? '') === strtolower($requestedColor);
            });
        }
        // If requested color not found or not provided, default to the first item.
        if (!$initialColorItem && !empty($galleryImages)) {
            $initialColorItem = $galleryImages[0];
        }

        $productGroup = collect($galleryImages)->map(function ($galleryItem) use ($product, $galleryImages) {
                return [
                    'id' => $product->id, // All variants share the same product ID
                    'product_name' => e($product->product_name),
                    'color' => $galleryItem['color'] ?? 'Default',
                    'image' => Storage::url($galleryItem['image']),
                    'price' => $product->price,
                    'discount_price' => $product->discount_price,
                    'description' => e($product->description),
                    'serial_no' => e($product->serial_no),
                    'category_name' => $product->category->name ?? 'Uncategorized',
                    'category_slug' => $product->category->slug ?? '',
                    // The 'gallery' for each item is the full gallery of the parent product
                    'gallery' => collect($galleryImages)->map(fn($g) => ['image' => Storage::url($g['image']), 'color' => $g['color'] ?? 'Default'])->values()->all(),
                ];
            });
        return view('frontend.pages.show', compact('product', 'productGroup', 'initialColorItem'));
    }

    /**
     * Get product data for quick view modal.
     */
    public function quickView(Product $product)
    {
        $product->load('category');
        return response()->json([
            'id' => $product->id,
            'product_name' => $product->product_name,
            'description' => $product->description,
            'price_formatted' => number_format($product->price),
            'discount_price_formatted' => $product->discount_price > 0 ? number_format($product->discount_price) : null,
            'image_url' => $product->image ? Storage::url($product->image) : 'https://via.placeholder.com/400x400',
            'show_url' => route('products.show', $product->id),
            'serial_no' => $product->serial_no,
            'category' => $product->category,
        ]);
    }
    /**
     * Show the product verification form.
     */
    public function showVerifyForm()
    {
        return view('products.verify');
    }

    /**
     * Handle the product verification form submission.
     */
    public function handleVerification(Request $request)
    {
        $request->validate([
            'serial_no' => 'required|string|max:255',
        ]);

        $product = Product::where('serial_no', $request->serial_no)
                          ->orWhere('imei', $request->serial_no)
                          ->first();

        if (!$product) {
            Scan::create([
                'product_id' => null,
                'scan_data' => $request->serial_no,
                'source' => 'manual_verification',
                'status' => 'failed',
            ]);
            return redirect()->route('verify')
                ->withInput()
                ->with('error', 'Product not found. Please check the serial number and try again.');
        }

        // Ensure the product has a verification token
        if (empty($product->verification_token)) {
            $product->verification_token = Str::random(32);
            $product->save();
        }

        Scan::create([
            'product_id' => $product->id,
            'scan_data' => $request->serial_no,
            'source' => 'manual_verification',
            'status' => 'success',
        ]);

        // Redirect to a unique, verifiable URL
        return redirect()->route('products.verified', [
            'serial_no' => $product->serial_no,
            'token' => $product->verification_token,
        ]);
    }

    /**
     * Display the verified product page if the token is valid.
     */
    public function showVerifiedProduct(Request $request, $serial_no)
    {
        $token = $request->query('token');

        if (!$token) {
            return redirect()->route('verify')->with('error', 'Verification token is missing.');
        }

        $product = Product::where('serial_no', $serial_no)->first();

        // Check if product exists and token matches
        if (!$product || $product->verification_token !== $token) {
            return redirect()->route('verify')->with('error', 'Invalid verification link or product not found.');
        }

        // If valid, show the verification page with product details
        return view('products.verify', compact('product'));
    }

    /**
     * Show the form for editing the specified product.
     *
     * @param  \App\Models\Product  $product
     * @return \Illuminate\View\View
     */
    public function edit(Product $product)
    {
        return view('products.edit', compact('product'));
    }

    /**
     * Show the form for editing the specified product for an admin.
     *
     * @param  \App\Models\Product  $product
     * @return \Illuminate\View\View
     */
    public function adminEdit(Product $product)
    {
        return view('products.edit', compact('product'));
    }

    /**
     * Update the specified product in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Product  $product
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, Product $product)
    {
        $rules = [
            'product_name' => 'required|string|max:255',
            'serial_no' => 'nullable|string|max:255|unique:products,serial_no,' . $product->id,
            'imei' => 'nullable|string|max:255|unique:products,imei,' . $product->id,
            'condition' => 'sometimes|required|string|max:255',
            'status' => 'sometimes|required|string|max:255',
            'quality' => 'sometimes|required|string|max:255',
            'color' => 'sometimes|required|array', // Expect an array of colors
            'distributor_name' => 'nullable|string|max:255',
            'retailer_name' => 'nullable|string|max:255',
            'client_name' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            // Pricing fields
            'cost' => 'nullable|numeric|min:0',
            'price' => 'nullable|numeric|min:0',
            'retail_price' => 'nullable|numeric|min:0',
            'discount_price' => 'nullable|numeric|min:0|lte:price',
            'dealer_price' => 'nullable|numeric|min:0',
            'description' => 'nullable|string',
            'discount_type' => 'nullable|string|in:flat,percentage',
            'discount_value' => 'nullable|numeric|min:0',
            'gallery' => 'nullable|array',
            'gallery.*.color' => 'required_with:gallery|string|max:255',
            'gallery.*.image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',

        ];

        // For AJAX inline updates, we only validate the fields that are sent.
        if ($request->ajax()) {
            $validationRules = [];
            $updateFields = $request->except(['_method', '_token', 'product_id', 'redirect_to']);

            foreach ($updateFields as $field => $value) {
                if (isset($rules[$field])) {
                    $validationRules[$field] = $rules[$field];
                }
            }
        } else {
            $validationRules = $rules;
        }
        $request->validate($validationRules);

        $updateData = $request->except(['gallery', '_token', '_method', 'discount_type', 'discount_value']);

        // --- Handle Gallery Images ---
        $currentGallery = $product->gallery_images ?? [];
        $processedGallery = [];
        $submittedColors = $request->input('color', []);
        $processedColors = [];

        // 1. Process all submitted gallery items (new uploads and existing ones)
        if ($request->has('gallery') && is_array($request->gallery)) {
            foreach ($request->gallery as $index => $item) {
                $color = $item['color'];

                // Only process items for colors that are actually selected
                if (in_array($color, $submittedColors)) {
                    // A. If a new file is uploaded for this color
                    if ($request->hasFile("gallery.{$index}.image")) {
                        $file = $request->file("gallery.{$index}.image");
                        $filename = preg_replace('/[^A-Za-z0-9\._-]/', '', basename($file->getClientOriginalName()));
                        $imagePath = $file->storeAs('products', $filename, 'public');

                        // Delete the old image for this color if it exists
                        $oldItem = collect($currentGallery)->firstWhere('color', $color);
                        if ($oldItem && isset($oldItem['image']) && $oldItem['image'] !== $imagePath) {
                            Storage::disk('public')->delete($oldItem['image']);
                        }
                        $processedGallery[] = ['color' => $color, 'image' => $imagePath];
                    }
                    // B. If no new file, but an existing image path is provided
                    elseif (isset($item['existing']) && !empty($item['existing'])) {
                        $imagePath = $item['existing'];
                        $processedGallery[] = ['color' => $color, 'image' => $imagePath];
                    }
                    $processedColors[] = $color;
                }
            }
        }

        // 2. Keep images for selected colors that were not part of the form submission (e.g., no new image uploaded)
        $unmodifiedOldImages = collect($currentGallery)->filter(function ($item) use ($submittedColors, $processedColors) {
            return in_array($item['color'], $submittedColors) && !in_array($item['color'], $processedColors);
        });

        $updateData['gallery_images'] = collect($processedGallery)->merge($unmodifiedOldImages)->values()->all();

        // Set the main product image to the first image in the *filtered* gallery
        if (!empty($updateData['gallery_images'])) {
            $updateData['image'] = $updateData['gallery_images'][0]['image'];
        } else {
            // If all images were removed, clear the main image field as well
            $updateData['image'] = null; // Explicitly set to null
        }

        $product->fill($updateData);

        $product->save();


        // Handle inline AJAX updates if requested
        if ($request->ajax()) {
            // Add image_url to the response for AJAX updates
            $product->image_url = $product->image ? Storage::url($product->image) : 'https://via.placeholder.com/150';

            // Recalculate stats and include them in the AJAX response
            $statsQuery = Product::query();
            $stats = [
                'total_products' => $statsQuery->clone()->count(),
                'in_warehouse'   => $statsQuery->clone()->where('status', 'In Warehouse')->count(),
                'shipped'        => $statsQuery->clone()->where('status', 'Shipped')->count(),
                'delivered'      => $statsQuery->clone()->where('status', 'Delivered')->count(),
                'sold'           => $statsQuery->clone()->where('status', 'Sold')->count(),
                'returned'       => $statsQuery->clone()->where('status', 'Returned')->count(),
            ];
            return response()->json([
                'success' => true,
                'message' => 'Product updated successfully.',
                'product' => $product,
                'stats' => $stats]);
        }

        // Redirect back to the intended page (e.g., stock index or products list)
        $redirectRoute = $request->input('redirect_to', 'admin.stock.index');
        return redirect()->route($redirectRoute)->with('success', 'Product updated successfully.');
    }

    /**
     * Show the form for creating a new product.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        // This method returns the view that contains the form to add a new product.
        return view('admin.products.create');
    }

    /**
     * Store a newly created product in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'product_name' => 'required|string|max:255',
            'serial_no' => 'nullable|string|max:255|unique:products,serial_no,NULL,id',
            'imei' => 'nullable|string|max:255|unique:products,imei',
            'condition' => 'nullable|string|max:255',
            'status' => 'nullable|string|max:255',
            'quality' => 'nullable|string|max:255',
            'color' => 'nullable|array', // Making color optional for scanner creation
            'distributor_name' => 'nullable|string|max:255',
            'distributor_code' => 'nullable|string|max:255',
            'retailer_name' => 'nullable|string|max:255',
            'client_name' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'cost' => 'nullable|numeric|min:0',
            'price' => 'nullable|numeric|min:0',
            'retail_price' => 'nullable|numeric|min:0',
            'discount_price' => 'nullable|numeric|min:0|lte:price',
            'dealer_price' => 'nullable|numeric|min:0',
            'discount_type' => 'nullable|string|in:flat,percentage',
            'discount_value' => 'nullable|numeric|min:0',
            'gallery' => 'nullable|array',
            'gallery.*.color' => 'required_with:gallery|string|max:255',
            'gallery.*.image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
        ]);

        // Exclude gallery from initial creation to handle it separately
        $productData = collect($validatedData)->except('gallery')->all();
        $product = new Product($productData);

        // If distributor_name is not provided, set a default value.
        if (empty($product->distributor_name)) {
            $product->distributor_name = 'N/A';
        }
        $product->created_by = auth()->user()->name ?? 'Web Form';

        // Handle Color-wise Image Gallery
        $galleryData = [];
        if ($request->has('gallery')) {
            foreach ($request->gallery as $index => $item) {
                // Only process items that have an image upload
                if ($request->hasFile("gallery.{$index}.image")) {
                    $file = $request->file("gallery.{$index}.image");
                    // Use original filename. Sanitize it to prevent directory traversal issues.
                    $filename = preg_replace('/[^A-Za-z0-9\._-]/', '', basename($file->getClientOriginalName()));
                    $imagePath = $file->storeAs('products', $filename, 'public');
                    $color = $item['color'];
                    $galleryData[] = ['color' => $color, 'image' => $imagePath];
                }
            }
        }

        // Set the main product image to the first image in the gallery
        if (!empty($galleryData)) {
            $product->image = $galleryData[0]['image'];
            $product->gallery_images = $galleryData;
        } else {
            $product->image = null;
            $product->gallery_images = []; // Explicitly set to empty array
        }

        // The 'color' field from the form is an array. We'll store it as JSON.
        $product->color = $validatedData['color'] ?? ['Unknown'];

        if (empty($product->distributor_code)) {
            $product->distributor_code = 'D-SYS-' . strtoupper(substr(str_shuffle("0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"), 0, 6));
        }

        // Auto-generate a unique serial number if it's not provided
        if (empty($product->serial_no)) {
            do {
                $product->serial_no = 'PROD-' . strtoupper(Str::random(10));
            } while (Product::where('serial_no', $product->serial_no)->exists());
        }

        $product->save();

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'New product created successfully.',
                'product' => $product,
            ]);
        }

        // Redirect to the stock index page with a success message.
        return redirect()->route('admin.stock.index')->with('success', 'New product created successfully.');
    }
    /**
     * Generate PDF report - PUBLIC ACCESS
     */
    public function generateReport($id)
    {
        try {
            $product = Product::findOrFail($id);

            $pdf = Pdf::loadView('pdf.report', compact('product'));

            return $pdf->download('product-report-' . $product->serial_no . '.pdf');
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            // If product is not found, redirect to home with an error message.
            return redirect()->route('home')->with('error', 'Verification not found. The requested product does not exist.');
        }
    }

    /**
     * Display sticker printing page
     */
    public function stickerPrint(Request $request)
    {
        $serialNo = $request->get('serial_no');
        $quantity = $request->get('quantity', 40); // Default to 40 (one page)

        $query = Product::query();

        // If a starting serial number is provided, fetch products from that point.
        if ($serialNo) {
            $query->where('serial_no', '>=', $serialNo);
        }

        // Fetch the requested number of products, ordered by serial number.
        $products = $query->orderBy('serial_no', 'asc')->take($quantity)->get();


        return view('products.sticker-print', compact('serialNo', 'quantity', 'products'));
    }

    /**
     * Admin products management - ADMIN ONLY
     */
    public function adminIndex(Request $request)
    {
        $search = $request->get('search');
        $quality = $request->get('quality');
        $color = $request->get('color');

        $query = Product::query();

        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('serial_no', 'like', "%{$search}%")
                  ->orWhere('imei', 'like', "%{$search}%")
                  ->orWhere('product_name', 'like', "%{$search}%");
            });
        }

        if ($quality) {
            $query->where('quality', $quality);
        }

        if ($color) {
            $query->where('color', $color);
        }

        $products = $query->orderBy('created_at', 'desc')->paginate(20);

        // Calculate statistics for the cards
        $stats = [
            'total_products' => Product::count(),
            'in_warehouse'   => Product::where('status', 'In Warehouse')->count(),
            'shipped'        => Product::where('status', 'Shipped')->count(),
            'delivered'      => Product::where('status', 'Delivered')->count(),
            'sold'           => Product::where('status', 'Sold')->count(),
            'returned'       => Product::where('status', 'Returned')->count(),
        ];

        return view('products.list', compact('products', 'search', 'quality', 'color', 'stats'));
    }

    /**
     * Display stock management page - ADMIN ONLY
     */
    public function stockIndex(Request $request)
    {
        $search = $request->get('search');
        $status = $request->get('status');
        $productName = $request->get('product_name');
        $color = $request->get('color');

        $query = Product::query();

        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('serial_no', 'like', "%{$search}%")
                  ->orWhere('imei', 'like', "%{$search}%")
                  ->orWhere('product_name', 'like', "%{$search}%");
            });
        }

        if ($status && $status !== 'all') {
            $query->where('status', $status);
        }

        if ($productName && $productName !== 'all') {
            $query->where('product_name', $productName);
        }

        if ($color && $color !== 'all') {
            $query->where('color', $color);
        }

        // Fetch all unique product names and colors for the filter dropdowns
        $productNamesCollection = Product::query()
            ->whereNotNull('product_name')
            ->distinct()
            ->orderBy('product_name')
            ->pluck('product_name');

        $productNames = $productNamesCollection->groupBy(function ($item) {
            return explode(' ', $item, 2)[0]; // Group by the first word (brand)
        });
        $rawColors = Product::query()
            ->whereNotNull('color')
            ->distinct()
            ->pluck('color');

        $productColors = $rawColors->flatMap(function ($color) {
            // $color can be a string or an array due to casting
            return is_array($color) ? $color : [$color];
        })->unique()->sort()->values();


        $products = $query->orderBy('created_at', 'desc')->paginate(20)->withQueryString();

        // Create a separate query for stats that is not paginated
        $statsQuery = Product::query();
        // Calculate statistics for the cards
        $stats = [
            'total_products' => $statsQuery->clone()->count(),
            'in_warehouse'   => $statsQuery->clone()->where('status', 'In Warehouse')->count(),
            'shipped'        => $statsQuery->clone()->where('status', 'Shipped')->count(),
            'delivered'      => $statsQuery->clone()->where('status', 'Delivered')->count(),
            'sold'           => $statsQuery->clone()->where('status', 'Sold')->count(),
            'returned'       => $statsQuery->clone()->where('status', 'Returned')->count(),
        ];

        if ($request->ajax()) {
            return response()->json([
                'table_html' => view('admin.partials.product_table_body', compact('products', 'productNames', 'productColors'))->render(),
                'pagination_html' => $products->links('vendor.pagination.tailwind')->toHtml(),
                'stats' => $stats,
            ]);
        }

        return view('admin.index', compact('products', 'stats', 'productNames', 'productColors'));
    }

    /**
     * Show the form for editing the specified product for stock management.
     *
     * @param  \App\Models\Product  $product
     * @return \Illuminate\View\View
     */
    public function stockEdit(Product $product)
    {
        return view('admin.stock.edit', compact('product'));
    }

    /**
     * Display scanner page - ADMIN ONLY
     */
    public function scannerIndex()
    {
        // Fetch all unique product names for the datalist suggestion
        $productNames = Product::query()
            ->whereNotNull('product_name')
            ->distinct()
            ->orderBy('product_name')
            ->pluck('product_name');

        return view('admin.scanner.index', compact('productNames'));
    }



    /**
     * Delete product - ADMIN ONLY
     */
    public function destroy($id)
    {
        $product = Product::findOrFail($id);
        $product->delete();

        return redirect()->route('admin.products')
            ->with('success', 'Product deleted successfully');
    }
}
