@extends('layouts.app')

@section('title', 'Add New Product')

@section('content')
    <div class="container mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <div class="max-w-4xl mx-auto bg-white rounded-lg shadow-md overflow-hidden">
            <div class="bg-gradient-to-r from-blue-600 to-indigo-700 px-4 sm:px-6 py-4">
                <h1 class="text-2xl font-bold text-white flex items-center"><i class="fas fa-plus-circle mr-3"></i>Add New
                    Product</h1>
                <p class="text-blue-100 mt-1">Enter the details for a new product to add to your inventory.</p>
            </div>

            <form action="{{ route('admin.products.store') }}" method="POST" enctype="multipart/form-data" class="p-4 sm:p-6">
                @csrf

                @if ($errors->any())
                    <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 mb-6" role="alert">
                        <p class="font-bold">Please correct the errors below:</p>
                        <ul>
                            @foreach ($errors->all() as $error)
                                <li>{{ $error }}</li>
                            @endforeach
                        </ul>
                    </div>
                @endif

                <div class="grid grid-cols-1 md:grid-cols-2 gap-6" x-data="{
                    selectedColors: {{ json_encode(old('color', [])) ?? '[]' }},
                    mrp: parseFloat('{{ old('price', 0) }}'),
                    discountType: '{{ old('discount_type', 'flat') }}',
                    discountValue: parseFloat('{{ old('discount_value', 0) }}'),
                    get dp() {
                        if (this.discountType === 'flat') {
                            return this.mrp > 0 ? this.mrp - this.discountValue : 0;
                        }
                        if (this.discountType === 'percentage') {
                            return this.mrp > 0 ? this.mrp - (this.mrp * this.discountValue / 100) : 0;
                        }
                        return 0;
                    },
                    availableColors: ['Black', 'White', 'Silver', 'Gray', 'Red', 'Blue', 'Green', 'Gold', 'Cyan', 'Pink', 'Other'],
                    toggleColor(color) {
                        this.selectedColors.includes(color) ? this.selectedColors = this.selectedColors.filter(c => c !== color) : this.selectedColors.push(color)
                    },
                    init() {
                        // When discount type changes, reset the value to avoid miscalculation
                        this.$watch('discountType', () => this.discountValue = 0);
                    }
                }">
                    <!-- Product Name (Brand & Model) -->
                    <div>
                        <label for="product_name" class="block text-sm font-medium text-gray-700">Product Name (Brand &
                            Model)</label>
                        <input type="text" name="product_name" id="product_name" value="{{ old('product_name') }}"
                            class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500"
                            required>
                    </div>

                    <!-- Color -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700">Colors</label>
                        <div class="mt-2 flex flex-wrap gap-2">
                            <template x-for="color in availableColors" :key="color">
                                <button type="button" @click="toggleColor(color)"
                                    :class="selectedColors.includes(color) ? 'bg-blue-600 text-white border-blue-600' :
                                        'bg-white text-gray-700 border-gray-300 hover:bg-gray-50'"
                                    class="px-3 py-1 border rounded-full text-sm font-medium transition-colors focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                                    x-text="color">
                                </button>
                            </template>
                        </div>
                        <!-- Hidden inputs to submit selected colors -->
                        <template x-for="color in selectedColors" :key="color"><input type="hidden"
                                name="color[]" :value="color"></template>
                    </div>

                    <!-- Purchase Price (PP) -->
                    <div>
                        <label for="cost" class="block text-sm font-medium text-gray-700">Purchase Price (PP)</label>
                        <input type="number" name="cost" id="cost" value="{{ old('cost') }}"
                            class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500"
                            step="0.01" min="0" placeholder="e.g., 50000.00">
                    </div>

                    <!-- Maximum Retail Price (MRP) -->
                    <div>
                        <label for="price" class="block text-sm font-medium text-gray-700">Maximum Retail Price
                            (MRP)</label>
                        <input type="number" name="price" id="price" x-model.number="mrp"
                            class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500"
                            step="0.01" min="0" placeholder="e.g., 65000.00">
                    </div>

                    <!-- Retail Price (RP) -->
                    <div>
                        <label for="retail_price" class="block text-sm font-medium text-gray-700">RP (Retail Price)</label>
                        <input type="number" name="retail_price" id="retail_price" value="{{ old('retail_price') }}"
                            class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500"
                            step="0.01" min="0" placeholder="e.g., 60000.00">
                    </div>
                    <!-- Dealer Price (DP)-->
                    <div>
                        <label for="dealer_price" class="block text-sm font-medium text-gray-700">Dealer Price(DP)</label>
                        <input type="number" name="dealer_price" id="dealer_price" value="{{ old('dealer_price') }}"
                            class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500"
                            step="0.01" min="0" placeholder="e.g., 55000.00">
                    </div>


                    <!-- Discount Section -->
                    <div class="md:col-span-2 grid grid-cols-1 md:grid-cols-3 gap-6 border-t pt-6">
                        <!-- Discount Type -->
                        <div>
                            <label for="discount_type" class="block text-sm font-medium text-gray-700">Discount Type</label>
                            <select name="discount_type" id="discount_type" x-model="discountType"
                                class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500">
                                <option value="flat">Flat Amount (৳)</option>
                                <option value="percentage">Percentage (%)</option>
                            </select>
                        </div>

                        <!-- Discount Value -->
                        <div>
                            <label for="discount_value" class="block text-sm font-medium text-gray-700">Discount
                                Value</label>
                            <input type="number" name="discount_value" id="discount_value" x-model.number="discountValue"
                                class="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500"
                                step="0.01" min="0"
                                :placeholder="discountType === 'flat' ? 'e.g., 2000' : 'e.g., 10'">
                        </div>

                        <!-- Calculated Discount Price -->
                        <div>
                            <label for="discount_price" class="block text-sm font-medium text-gray-700">Calculated Discount
                                Price</label>
                            <input type="number" name="discount_price" id="discount_price"
                                :value="dp > 0 ? dp.toFixed(2) : 0"
                                class="mt-1 block w-full border-gray-300 rounded-md shadow-sm bg-gray-100" readonly>
                            <div x-show="discountValue > 0 && mrp > 0 && dp < mrp" class="mt-2 text-sm text-green-600">
                                <span
                                    x-text="discountType === 'flat' ? 'Discount: ৳' + discountValue.toFixed(2) : 'Discount: ' + discountValue + '%'"></span>
                                (Final: ৳<span x-text="dp.toFixed(2)"></span>)
                            </div>
                            <div x-show="dp >= mrp && mrp > 0 && discountValue > 0" class="mt-2 text-sm text-red-600">
                                Discount makes price free or negative.
                            </div>
                        </div>
                    </div>


                    <!-- Image Upload -->
                    <div class="md:col-span-2" x-data="imageColorUploader({
                        allColors: {
                            'Black': 'bg-black',
                            'White': 'bg-white border',
                            'Silver': 'bg-slate-300',
                            'Gray': 'bg-gray-500',
                            'Red': 'bg-red-500',
                            'Blue': 'bg-blue-500',
                            'Green': 'bg-green-500',
                            'Gold': 'bg-yellow-400',
                            'Cyan': 'bg-cyan-400',
                            'Pink': 'bg-pink-400',
                            'Other': 'bg-gray-200'
                        },
                    })">
                        <label class="block text-sm font-medium text-gray-700">Product Images by Color</label>
                        <p class="text-xs text-gray-500 mt-1">Select colors in the 'Colors' field above, then upload an
                            image for each. The first image will be the main product image.</p>

                        <div class="mt-4 grid grid-cols-3 sm:grid-cols-4 md:grid-cols-5 lg:grid-cols-6 gap-4">
                            <template x-for="(colorName, index) in selectedColors" :key="colorName">
                                <div class="flex flex-col items-center gap-2">
                                    <!-- Color Swatch and Image Preview -->
                                    <label :for="'file_input_' + colorName"
                                        class="relative w-20 h-20 rounded-full cursor-pointer group flex items-center justify-center border-2"
                                        :class="colorImages[colorName] ? 'border-blue-500' : 'border-dashed border-gray-300'">
                                        <img x-if="colorImages[colorName]?.preview" :src="colorImages[colorName].preview"
                                            class="w-full h-full object-cover rounded-full">
                                        <div x-else :class="allColors[colorName] || 'bg-gray-200'"
                                            class="w-full h-full rounded-full"></div>

                                        <!-- Upload/Change Icon -->
                                        <div
                                            class="absolute inset-0 bg-black bg-opacity-50 rounded-full flex items-center justify-center opacity-0 group-hover:opacity-100 transition-opacity">
                                            <i class="fas fa-camera text-white text-xl"></i>
                                        </div>
                                    </label>

                                    <!-- Color Name and Remove Button -->
                                    <div class="flex items-center gap-2">
                                        <span class="text-sm font-medium text-gray-700" x-text="colorName"></span>
                                        <button x-show="colorImages[colorName]" @click.prevent="removeImage(colorName)"
                                            type="button" class="text-red-500 hover:text-red-700 text-xs"
                                            title="Remove Image">
                                            &times;
                                        </button>
                                    </div>

                                    <!-- Hidden Inputs for Form Submission -->
                                    <input type="file" :id="'file_input_' + colorName"
                                        :name="`gallery[${index}][image]`" @change="handleFileChange($event, colorName)"
                                        class="hidden">
                                    <input type="hidden" :name="`gallery[${index}][color]`" :value="colorName">
                                </div>
                            </template>
                        </div>
                    </div>
                </div>

                <div
                    class="mt-8 flex flex-col-reverse sm:flex-row sm:justify-end sm:space-x-4 space-y-4 space-y-reverse sm:space-y-0">
                    <a href="{{ route('admin.stock.index') }}"
                        class="w-full sm:w-auto justify-center text-center bg-secondary text-white px-6 py-2 rounded-lg hover:bg-opacity-90 transition-colors">
                        Cancel
                    </a>
                    <button type="submit"
                        class="w-full sm:w-auto justify-center bg-primary text-white px-6 py-2 rounded-lg hover:bg-opacity-90 transition-colors flex items-center">
                        <i class="fas fa-plus-circle mr-2"></i>Add Product
                    </button>
                </div>
            </form>
        </div>
    </div>
@endsection

@push('scripts')
    <script>
        // Initialize TomSelect for the color dropdown
        function imageColorUploader({
            allColors
        }) {
            return {
                allColors: allColors,
                // selectedColors is inherited from the parent x-data scope
                colorImages: {},

                init() {},

                handleFileChange(event, colorName) {
                    const file = event.target.files[0];
                    if (file) {
                        this.colorImages[colorName] = {
                            ...this.colorImages[colorName], // Preserve existing data if any
                            preview: URL.createObjectURL(file)
                        };
                        this.$forceUpdate(); // Force Alpine to re-render the component
                    }
                },

                removeImage(colorName) {
                    this.colorImages[colorName] = undefined;
                    document.getElementById('file_input_' + colorName).value = ''; // Clear the file input
                }
            }
        }
    </script>
@endpush
