<!-- Quick View Modal -->
<div
    x-data="{
        open: false,
        loading: false,
        product: null,
        inWishlist: false,
        inCompare: false,
        isAddingToCart: false,
        quantity: 1,
        fetchProduct(id) {
            this.open = true;
            this.loading = true;
            this.product = null;
            this.inWishlist = false;
            this.inCompare = false;

            fetch(`/products/quick-view/${id}`)
                .then(response => response.json())
                .then(data => {
                    this.product = data;
                    this.inCompare = data.in_compare;
                    this.inWishlist = data.in_wishlist;
                    this.quantity = 1; // Reset quantity on new product
                })
                .catch(error => {
                    console.error('Error fetching product:', error);
                    this.open = false;
                })
                .finally(() => {
                    this.loading = false;
                });
        },
        increment() {
            this.quantity++;
        },
        decrement() {
            if (this.quantity > 1) this.quantity--;
        },
        toggleWishlist() {
            if (!this.product) return;
            this.inWishlist = !this.inWishlist; // Optimistic update

            fetch(`/wishlist/toggle/${this.product.id}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name=\'csrf-token\']').getAttribute('content'),
                    'Accept': 'application/json',
                }
            })
            .then(response => {
                if (response.status === 401) {
                    window.location.href = '{{ route('login') }}';
                    return Promise.reject('Unauthenticated');
                }
                return response.json();
            })
            .then(data => {
                if (data.success) {
                    window.dispatchEvent(new CustomEvent('show-notification', { detail: { message: data.message, type: 'success' }}));
                    const wishlistCountEl = document.getElementById('wishlist-count');
                    if (wishlistCountEl) wishlistCountEl.textContent = data.wishlistCount;
                } else {
                    this.inWishlist = !this.inWishlist; // Revert on failure
                }
            })
            .catch(error => { if (error !== 'Unauthenticated') this.inWishlist = !this.inWishlist; });
        },
        addToCart() {
            if (!this.product) return;
            this.isAddingToCart = true;

            fetch(`/cart/add/${this.product.id}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name=\'csrf-token\']').getAttribute('content'),
                    'Accept': 'application/json',
                },
                body: JSON.stringify({ quantity: this.quantity })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    window.dispatchEvent(new CustomEvent('show-notification', { detail: { message: `Added '${this.product.product_name}' to cart!`, type: 'success' }}));
                    const cartCountEl = document.getElementById('cart-count');
                    if (cartCountEl) cartCountEl.textContent = data.cartCount;
                    this.open = false; // Close modal on success
                } else {
                    window.dispatchEvent(new CustomEvent('show-notification', { detail: { message: data.message || 'Could not add to cart.', type: 'error' }}));
                }
            })
            .finally(() => { this.isAddingToCart = false; });
        },
        addToCompare() {
            if (!this.product) return;
            this.inCompare = true; // Optimistic update

            fetch(`/compare/add/${this.product.id}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name=\'csrf-token\']').getAttribute('content'),
                    'Accept': 'application/json',
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    window.dispatchEvent(new CustomEvent('show-notification', { detail: { message: data.message, type: 'success' }}));
                    const compareCountEl = document.getElementById('compare-count');
                    if (compareCountEl) compareCountEl.textContent = data.compareCount;
                } else {
                    this.inCompare = false; // Revert on failure
                    window.dispatchEvent(new CustomEvent('show-notification', { detail: { message: data.message, type: 'error' }}));
                }
            })
            .catch(() => { this.inCompare = false; });
        }
    }"
    @keydown.escape.window="open = false"
    @open-quick-view.window="fetchProduct($event.detail.id)"
>
    <div
        x-show="open"
        x-transition:enter="transition ease-out duration-300"
        x-transition:enter-start="opacity-0"
        x-transition:enter-end="opacity-100"
        x-transition:leave="transition ease-in duration-300"
        x-transition:leave-start="opacity-100"
        x-transition:leave-end="opacity-0"
        class="fixed inset-0 bg-gray-900 bg-opacity-75 flex items-start md:items-center justify-center z-50 p-4"
        style="display: none;"
    >
        <!-- Modal Content -->
        <div
            x-show="open"
            x-transition:enter="transition ease-out duration-300"
            x-transition:enter-start="opacity-0 transform scale-90"
            x-transition:enter-end="opacity-100 transform scale-100"
            x-transition:leave="transition ease-in duration-200"
            x-transition:leave-start="opacity-100 transform scale-100"
            x-transition:leave-end="opacity-0 transform scale-90"
            @click.away="open = false"
            class="bg-white rounded-2xl shadow-2xl w-full max-w-4xl max-h-[90vh] overflow-y-auto"
        >
            <template x-if="loading">
                <div class="p-8 text-center">
                    <i class="fas fa-spinner fa-spin text-4xl text-blue-500"></i>
                    <p class="mt-4 text-gray-600">Loading product details...</p>
                </div>
            </template>

            <template x-if="!loading && product">
                <div class="grid grid-cols-1 md:grid-cols-2 gap-8 p-6 md:p-8 relative">
                    <!-- Close button -->
                    <button @click="open = false" class="absolute top-4 right-4 text-gray-400 hover:text-gray-600 transition">
                        <i class="fas fa-times fa-lg"></i>
                    </button>

                    <!-- Image -->
                    <div class="flex items-center justify-center bg-gray-100 rounded-xl p-4">
                        <img :src="product.image_url" :alt="product.product_name" class="max-h-64 md:max-h-96 w-full object-contain">
                    </div>

                    <!-- Details -->
                    <div class="flex flex-col">
                        <h2 class="text-2xl lg:text-3xl font-bold text-gray-900 mb-3" x-text="product.product_name"></h2>

                        <!-- Stock Status & Category -->
                        <div class="flex items-center gap-4 mb-4">
                            <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-semibold bg-green-100 text-green-800">
                                <i class="fas fa-check-circle mr-2"></i> In Stock
                            </span>
                            <a href="#" class="text-sm text-blue-600 hover:underline" x-text="product.category ? product.category.name : 'Uncategorized'"></a>
                        </div>

                        <!-- Price -->
                        <div class="mb-5">
                            <template x-if="product.discount_price_formatted">
                                <div>
                                    <span class="text-3xl font-bold text-blue-600" x-text="`৳${product.discount_price_formatted}`"></span>
                                    <span class="text-lg text-gray-500 line-through ml-3" x-text="`৳${product.price_formatted}`"></span>
                                </div>
                            </template>
                            <template x-if="!product.discount_price_formatted">
                                <span class="text-3xl font-bold text-blue-600" x-text="`৳${product.price_formatted}`"></span>
                            </template>
                        </div>

                        <!-- Short Description -->
                        <p class="text-gray-600 mb-6 leading-relaxed text-sm" x-html="product.description ? (product.description.length > 150 ? product.description.substring(0, 150) + '...' : product.description) : 'No description available.'"></p>

                        <div class="mt-auto pt-6 border-t">
                            <!-- Quantity and Add to Cart -->
                            <div class="flex items-center gap-4 mb-4">
                                <!-- Quantity Input -->
                                <div class="flex items-center border border-gray-300 rounded-lg">
                                    <button @click="decrement()" class="px-3 py-2 text-gray-600 hover:bg-gray-100 rounded-l-lg transition">-</button>
                                    <input type="text" x-model="quantity" class="w-12 text-center border-0 focus:ring-0" readonly>
                                    <button @click="increment()" class="px-3 py-2 text-gray-600 hover:bg-gray-100 rounded-r-lg transition">+</button>
                                </div>

                                <button @click.prevent="addToCart()" :disabled="isAddingToCart" class="flex-1 v-btn v-btn-primary">
                                    <template x-if="!isAddingToCart">
                                        <span><i class="fas fa-shopping-cart mr-2"></i> Add to Cart</span>
                                    </template>
                                    <template x-if="isAddingToCart">
                                        <span><i class="fas fa-spinner fa-spin mr-2"></i> Adding...</span>
                                    </template>
                                </button>
                            </div>

                            <!-- Wishlist & Compare -->
                            <div class="flex items-center gap-6 text-sm">
                                <button @click.prevent="toggleWishlist()" class="text-gray-600 hover:text-blue-600 transition" :class="{ 'text-red-500 font-semibold': inWishlist }">
                                    <i class="mr-2 fa-heart" :class="inWishlist ? 'fas' : 'far'"></i>
                                    <span x-text="inWishlist ? 'Added to Wishlist' : 'Add to Wishlist'"></span>
                                </button>
                                <button @click.prevent="addToCompare()" class="text-gray-600 hover:text-blue-600 transition" :class="{ 'text-blue-500 font-semibold': inCompare }">
                                    <i class="fas fa-exchange-alt mr-2"></i> Compare
                                    <span x-text="inCompare ? 'Added to Compare' : 'Add to Compare'"></span>
                                </button>
                            </div>

                            <!-- Meta -->
                            <div class="text-xs text-gray-500 space-y-1 mt-6">
                                <p><span class="font-semibold text-gray-700">SKU:</span> <span x-text="product.serial_no || 'N/A'"></span></p>
                            </div>

                            <a :href="product.show_url" class="inline-block mt-4 text-blue-600 font-semibold hover:underline text-sm">
                                View Full Product Details <i class="fas fa-arrow-right ml-1 text-xs"></i>
                            </a>
                        </div>
                    </div>
                </div>
            </template>
        </div>
    </div>
</div>
