<?php

namespace App\Http\Controllers\Frontend;

use App\Models\Product;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class CartController extends Controller
{
    /**
     * Display the cart page.
     */
    public function index(Request $request)
    {
        $cart = $request->session()->get('cart', []);
        $total = 0;

        foreach ($cart as $id => $details) {
            $price = $details['discount_price'] ?? $details['price'];
            $total += $price * $details['quantity'];
        }

        return view('frontend.pages.cart', compact('cart', 'total'));
    }

    /**
     * Add a product to the cart.
     */
    public function add(Request $request, Product $product)
    {
        $cart = $request->session()->get('cart', []);

        if (isset($cart[$product->id])) {
            $cart[$product->id]['quantity']++;
        } else {
            $cart[$product->id] = [
                "product_name" => $product->product_name,
                "quantity" => 1,
                "price" => $product->price,
                "discount_price" => $product->discount_price,
                "image" => $product->image,
            ];
        }

        $request->session()->put('cart', $cart);

        return response()->json([
            'success' => true,
            'message' => 'Product added to cart successfully!',
            'cartCount' => count($request->session()->get('cart', [])),
        ]);
    }

    /**
     * Update a product's quantity in the cart.
     */
    public function update(Request $request)
    {
        if ($request->id && $request->quantity) {
            $cart = $request->session()->get('cart');
            if (isset($cart[$request->id])) {
                $cart[$request->id]['quantity'] = $request->quantity;
                $request->session()->put('cart', $cart);

                // Recalculate total
                $total = 0;
                foreach ($cart as $id => $details) {
                    $price = $details['discount_price'] ?? $details['price'];
                    $total += $price * $details['quantity'];
                }

                return response()->json([
                    'success' => true,
                    'message' => 'Cart updated successfully.',
                    'subtotal' => number_format($total), // Send back updated totals
                    'total' => number_format($total)
                ]);
            }
        }
        return response()->json(['success' => false, 'message' => 'Could not update cart.'], 400); // 400 Bad Request
    }

    /**
     * Remove a product from the cart.
     */
    public function remove(Request $request)
    {
        if ($request->id) {
            $cart = $request->session()->get('cart');
            if (isset($cart[$request->id])) {
                unset($cart[$request->id]);
                $request->session()->put('cart', $cart);

                // Recalculate total
                $total = 0;
                foreach ($cart as $id => $details) {
                    $price = $details['discount_price'] ?? $details['price'];
                    $total += $price * $details['quantity'];
                }

                return response()->json([
                    'success' => true,
                    'message' => 'Product removed successfully.',
                    'subtotal' => number_format($total),
                    'total' => number_format($total),
                    'cartCount' => count($cart)
                ]);
            }
        }
        return response()->json(['success' => false, 'message' => 'Could not remove product.'], 400); // 400 Bad Request
    }
}
