<?php

namespace App\Http\Controllers;


use App\Models\Category;
use App\Models\Scan;
use Illuminate\Support\Str;
use App\Models\Product;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;

class ProductController extends Controller
{
    /**
     * Display a list of products for the public frontend.
     */
    public function index(Request $request)
    {
        $query = Product::query();

        // Handle search
        if ($request->filled('search')) {
            $search = $request->input('search');
            $query->where('product_name', 'like', "%{$search}%");
        }

        // Handle category filtering
        if ($request->filled('category')) {
            $categorySlugs = $request->input('category');
            if (!is_array($categorySlugs)) {
                $categorySlugs = [$categorySlugs];
            }

            $query->where(function ($q) use ($categorySlugs) {
                // Find categories and their children
                $categories = Category::with('children')->whereIn('slug', $categorySlugs)->get();
                $allCategoryIds = $categories->pluck('id');
                foreach ($categories as $category) {
                    $allCategoryIds = $allCategoryIds->merge($category->children->pluck('id'));
                }
                $q->whereIn('category_id', $allCategoryIds->unique());
            });
        }

        // Handle brand filtering
        if ($request->filled('brands')) {
            $brands = $request->input('brands');
            $query->where(function ($q) use ($brands) {
                if (!is_array($brands)) {
                    $brands = [$brands];
                }
                foreach ($brands as $brand) {
                    $q->orWhere('product_name', 'like', $brand . '%');
                }
            });
        }

        // Handle rating filtering
        if ($request->filled('ratings')) {
            // The 'rating' column does not exist in the 'products' table yet.
            // This filter is commented out to prevent errors.
            // To enable it, add a 'rating' column to your products table migration and seeder.
        }

        // Handle price range filtering
        if ($request->filled('price_min')) {
            $query->where('price', '>=', $request->input('price_min'));
        }
        if ($request->filled('price_max')) {
            $query->where('price', '<=', $request->input('price_max'));
        }

        $products = $query->with('category')->latest()->paginate(9);
        $maxPrice = Product::max('price');

        // For AJAX requests, we return the rendered product list and pagination
        if ($request->ajax()) {
            return response()->json([
                'products_html' => view('frontend.pages.partials.products-grid', compact('products'))->render(),
                'pagination_html' => $products->links()->toHtml(),
            ]);
        }

        return view('frontend.pages.products', compact('products', 'maxPrice'));
    }

    /**
     * Display a single product for the public frontend.
     */
    public function show(Product $product)
    {
        // Eager load category to prevent extra queries
        $product->load('category');

        return view('frontend.pages.show', compact('product'));
    }

    /**
     * Get product data for quick view modal.
     */
    public function quickView(Product $product)
    {
        $product->load('category');
        return response()->json([
            'id' => $product->id,
            'product_name' => $product->product_name,
            'description' => $product->description,
            'price_formatted' => number_format($product->price),
            'discount_price_formatted' => $product->discount_price ? number_format($product->discount_price) : null,
            'image_url' => $product->image ? asset('storage/' . $product->image) : 'https://via.placeholder.com/400x400',
            'show_url' => route('products.show', $product->id),
            'serial_no' => $product->serial_no,
            'category' => $product->category,
        ]);
    }
    /**
     * Show the product verification form.
     */
    public function showVerifyForm()
    {
        return view('products.verify');
    }

    /**
     * Handle the product verification form submission.
     */
    public function handleVerification(Request $request)
    {
        $request->validate([
            'serial_no' => 'required|string|max:255',
        ]);

        $product = Product::where('serial_no', $request->serial_no)
                          ->orWhere('imei', $request->serial_no)
                          ->first();

        if (!$product) {
            Scan::create([
                'product_id' => null,
                'scan_data' => $request->serial_no,
                'source' => 'manual_verification',
                'status' => 'failed',
            ]);
            return redirect()->route('verify')
                ->withInput()
                ->with('error', 'Product not found. Please check the serial number and try again.');
        }

        // Ensure the product has a verification token
        if (empty($product->verification_token)) {
            $product->verification_token = Str::random(32);
            $product->save();
        }

        Scan::create([
            'product_id' => $product->id,
            'scan_data' => $request->serial_no,
            'source' => 'manual_verification',
            'status' => 'success',
        ]);

        // Redirect to a unique, verifiable URL
        return redirect()->route('products.verified', [
            'serial_no' => $product->serial_no,
            'token' => $product->verification_token,
        ]);
    }

    /**
     * Display the verified product page if the token is valid.
     */
    public function showVerifiedProduct(Request $request, $serial_no)
    {
        $token = $request->query('token');

        if (!$token) {
            return redirect()->route('verify')->with('error', 'Verification token is missing.');
        }

        $product = Product::where('serial_no', $serial_no)->first();

        // Check if product exists and token matches
        if (!$product || $product->verification_token !== $token) {
            return redirect()->route('verify')->with('error', 'Invalid verification link or product not found.');
        }

        // If valid, show the verification page with product details
        return view('products.verify', compact('product'));
    }

    /**
     * Show the form for editing the specified product.
     *
     * @param  \App\Models\Product  $product
     * @return \Illuminate\View\View
     */
    public function edit(Product $product)
    {
        return view('products.edit', compact('product'));
    }

    /**
     * Update the specified product in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Product  $product
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, Product $product)
    {
        $rules = [
            'product_name' => 'required|string|max:255',
            'serial_no' => 'required|string|max:255|unique:products,serial_no,' . $product->id,
            'imei' => 'nullable|string|max:255|unique:products,imei,' . $product->id,
            'condition' => 'sometimes|required|string|max:255',
            'status' => 'sometimes|required|string|max:255',
            'quality' => 'sometimes|required|string|max:255',
            'color' => 'sometimes|required|string|max:255',
            'distributor_name' => 'nullable|string|max:255',
            'retailer_name' => 'nullable|string|max:255',
            'client_name' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048', // Add image validation
        ];

        // For AJAX inline updates, we only validate the fields that are sent.
        if ($request->ajax() && count($request->all()) === 1) {
            $field = array_key_first($request->all());
            $validationRules = [$field => $rules[$field] ?? 'sometimes|string|max:255'];
        } else {
            $validationRules = $rules;
        }
        $request->validate($validationRules);

        $product->fill($request->all());

        if ($request->hasFile('image')) {
            // Delete old image if it exists
            if ($product->image) { Storage::disk('public')->delete($product->image); }

            $file = $request->file('image');
            // Use original name as requested. Note: This can cause filename collisions if multiple files have the same name.
            $filename = $file->getClientOriginalName();
            $product->image = $file->storeAs('products', $filename, 'public');
        }
        $product->save();

        // Handle inline AJAX updates if requested
        if ($request->ajax()) {
            return response()->json(['success' => true, 'message' => 'Product updated successfully.', 'product' => $product]);
        }

        return redirect()->route('admin.products')->with('success', 'Product updated successfully.');
    }

    /**
     * Store a newly created product in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'product_name' => 'required|string|max:255',
            'serial_no' => 'required|string|max:255|unique:products,serial_no',
            'imei' => 'nullable|string|max:255|unique:products,imei',
            'condition' => 'required|string|max:255',
            'status' => 'required|string|max:255',
            'quality' => 'required|string|max:255',
            'color' => 'required|string|max:255',
            'distributor_name' => 'nullable|string|max:255',
            'distributor_code' => 'nullable|string|max:255',
            'retailer_name' => 'nullable|string|max:255',
            'client_name' => 'nullable|string|max:255',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
        ]);

        $product = new Product($validatedData);
        $product->scanned_by = auth()->user()->name ?? 'Web Scanner';
        $product->scanned_at = now();
        $product->created_by = auth()->user()->name ?? 'Web Scanner';

        if ($request->hasFile('image')) {
            $file = $request->file('image');
            // Use original name as requested. Note: This can cause filename collisions if multiple files have the same name.
            $filename = $file->getClientOriginalName();
            $product->image = $file->storeAs('products', $filename, 'public');
        }

        // Ensure distributor_code has a default value if not provided
        if (empty($product->distributor_code)) {
            $product->distributor_code = 'D-SYS-' . strtoupper(substr(str_shuffle("0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"), 0, 6));
        }
        $product->save();

        return response()->json([
            'success' => true,
            'message' => 'New product created successfully.',
            'product' => $product
        ]);
    }
    /**
     * Generate PDF report - PUBLIC ACCESS
     */
    public function generateReport($id)
    {
        try {
            $product = Product::findOrFail($id);

            $pdf = Pdf::loadView('pdf.report', compact('product'));

            return $pdf->download('product-report-' . $product->serial_no . '.pdf');
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            // If product is not found, redirect to home with an error message.
            return redirect()->route('home')->with('error', 'Verification not found. The requested product does not exist.');
        }
    }

    /**
     * Display sticker printing page
     */
    public function stickerPrint(Request $request)
    {
        $serialNo = $request->get('serial_no');
        $quantity = $request->get('quantity', 56);

        return view('products.sticker-print', compact('serialNo', 'quantity'));
    }

    /**
     * Admin products management - ADMIN ONLY
     */
    public function adminIndex(Request $request)
    {
        $search = $request->get('search');
        $quality = $request->get('quality');
        $color = $request->get('color');

        $query = Product::query();

        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('serial_no', 'like', "%{$search}%")
                  ->orWhere('imei', 'like', "%{$search}%")
                  ->orWhere('product_name', 'like', "%{$search}%");
            });
        }

        if ($quality) {
            $query->where('quality', $quality); // Fixed bug: was 'quality' string instead of $quality variable
        }

        if ($color) {
            $query->where('color', $color);
        }

        $products = $query->orderBy('created_at', 'desc')->paginate(20);

        // Calculate statistics for the cards
        $stats = [
            'total_products' => Product::count(),
            'in_warehouse'   => Product::where('status', 'In Warehouse')->count(),
            'shipped'        => Product::where('status', 'Shipped')->count(),
            'delivered'      => Product::where('status', 'Delivered')->count(),
            'sold'           => Product::where('status', 'Sold')->count(),
            'returned'       => Product::where('status', 'Returned')->count(),
        ];

        return view('products.list', compact('products', 'search', 'quality', 'color', 'stats'));
    }

    /**
     * Delete product - ADMIN ONLY
     */
    public function destroy($id)
    {
        $product = Product::findOrFail($id);
        $product->delete();

        return redirect()->route('admin.products')
            ->with('success', 'Product deleted successfully');
    }
}
