import './bootstrap';

// Global dropdown toggle function for product list
window.toggleDropdown = function(id) {
    const dropdownContent = document.getElementById(id);
    const dropdownBtn = dropdownContent.previousElementSibling;

    // Close all other open dropdowns
    document.querySelectorAll('.dropdown-content.show').forEach(openDropdown => {
        if (openDropdown.id !== id) {
            openDropdown.classList.remove('show');
            openDropdown.previousElementSibling.setAttribute('aria-expanded', 'false');
        }
    });

    dropdownContent.classList.toggle('show');
    const isExpanded = dropdownContent.classList.contains('show');
    dropdownBtn.setAttribute('aria-expanded', isExpanded);

    // Close dropdown if clicked outside
    if (isExpanded) {
        document.addEventListener('click', function closeDropdown(event) {
            if (!dropdownContent.contains(event.target) && !dropdownBtn.contains(event.target)) {
                dropdownContent.classList.remove('show');
                dropdownBtn.setAttribute('aria-expanded', 'false');
                document.removeEventListener('click', closeDropdown);
            }
        });
    }
};

// Global function to update product fields via AJAX
window.updateProductField = function(iteration, field, value) {
    const row = document.querySelector(`tbody tr:nth-child(${iteration})`);
    if (!row) return;

    const productId = row.getAttribute('data-product-id');
    const dropdownBtn = document.querySelector(`#${field}-btn-${iteration}`) || document.querySelector(`#status-badge-${iteration}`).parentElement;
    const originalContent = dropdownBtn.innerHTML;

    // Show loading indicator
    if (field !== 'status') {
        dropdownBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Updating...';
    } else {
        const badge = document.getElementById(`status-badge-${iteration}`);
        badge.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
    }

    fetch(`/admin/products/${productId}`, {
        method: 'PUT',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
            'X-Requested-With': 'XMLHttpRequest'
        },
        body: JSON.stringify({ [field]: value })
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('Network response was not ok');
        }
        return response.json();
    })
    .then(data => {
        if (data.success) {
            if (field === 'status') {
                const badge = document.getElementById(`status-badge-${iteration}`);
                badge.textContent = value;
                // Update badge color
                badge.className = 'px-2 inline-flex text-xs leading-5 font-semibold rounded-full ';
                const colorClasses = {
                    'In Warehouse': 'bg-blue-100 text-blue-800',
                    'Shipped': 'bg-yellow-100 text-yellow-800',
                    'Delivered': 'bg-green-100 text-green-800',
                    'Sold': 'bg-purple-100 text-purple-800',
                    'Returned': 'bg-red-100 text-red-800',
                };
                badge.classList.add(...(colorClasses[value] || 'bg-gray-100 text-gray-800').split(' '));
            } else {
                dropdownBtn.querySelector('span').textContent = value;
                dropdownBtn.innerHTML = `<span>${value}</span><i class="fas fa-chevron-down text-xs ml-1"></i>`;
            }
            window.showToast('success', 'Product updated successfully!');
        } else {
            window.showToast('error', data.message || 'Error updating product.');
            dropdownBtn.innerHTML = originalContent; // Restore on failure
        }
    })
    .catch(error => {
        console.error('Error:', error);
        window.showToast('error', 'An error occurred while updating the product.');
        dropdownBtn.innerHTML = originalContent; // Restore on error
    })
    .finally(() => {
        // Close the dropdown
        const dropdownContent = document.getElementById(`${field}-dropdown-${iteration}`);
        if (dropdownContent) dropdownContent.classList.remove('show');
    });
};

// Add ARIA attributes on page load for dropdowns
document.addEventListener('DOMContentLoaded', () => {
    document.querySelectorAll('.dropdown-btn').forEach(btn => {
        btn.setAttribute('aria-haspopup', 'true');
        btn.setAttribute('aria-expanded', 'false');
    });
    document.querySelectorAll('.dropdown-content').forEach(content => {
        content.setAttribute('role', 'menu');
    });
    document.querySelectorAll('.dropdown-content a').forEach(link => {
        link.setAttribute('role', 'menuitem');
    });
});

// Global toast notification function (to be used by scanner and product list)
window.showToast = function(type, message) {
    const toastContainer = document.getElementById('toast-container') || (() => {
        const div = document.createElement('div');
        div.id = 'toast-container';
        div.className = 'fixed top-4 right-4 z-50 space-y-2';
        document.body.appendChild(div);
        return div;
    })();

    const toast = document.createElement('div');
    toast.className = `p-4 rounded-lg shadow-lg flex items-center transition-all duration-300 transform translate-x-full opacity-0`;

    let bgColor, iconClass;
    if (type === 'success') {
        bgColor = 'bg-green-500';
        iconClass = 'fas fa-check-circle';
    } else if (type === 'error') {
        bgColor = 'bg-red-500';
        iconClass = 'fas fa-times-circle';
    } else if (type === 'warning') {
        bgColor = 'bg-yellow-500';
        iconClass = 'fas fa-exclamation-triangle';
    } else {
        bgColor = 'bg-blue-500';
        iconClass = 'fas fa-info-circle';
    }

    toast.innerHTML = `
        <div class="flex items-center text-white">
            <i class="${iconClass} mr-2"></i>
            <span>${message}</span>
            <button onclick="this.closest('.toast').remove()" class="ml-4 text-white hover:text-gray-200">
                <i class="fas fa-times"></i>
            </button>
        </div>
    `;
    toast.classList.add(bgColor, 'toast');
    toastContainer.appendChild(toast);

    // Animate in
    setTimeout(() => {
        toast.classList.remove('translate-x-full', 'opacity-0');
        toast.classList.add('translate-x-0', 'opacity-100');
    }, 100);

    // Auto-remove after 5 seconds
    setTimeout(() => {
        toast.classList.remove('translate-x-0', 'opacity-100');
        toast.classList.add('translate-x-full', 'opacity-0');
        toast.addEventListener('transitionend', () => toast.remove());
    }, 5000);
};
