// Dropdown functionality
function toggleDropdown(dropdownId) {
    const dropdown = document.getElementById(dropdownId);
    const isVisible = dropdown.classList.contains('show');

    // Close all other open dropdowns
    document.querySelectorAll('.dropdown-content.show').forEach(openDropdown => {
        openDropdown.classList.remove('show');
    });

    // If the clicked dropdown was not already visible, show it
    if (!isVisible) {
        dropdown.classList.add('show');
    }
}

// Update product field via AJAX
function updateProductField(rowIndex, field, value) {
    // Get the product ID from the table row's data attribute
    const productRow = document.querySelector(`tr[data-row-index='${rowIndex}']`);
    const productId = productRow.getAttribute('data-product-id');

    // Find the dropdown button to show a loading state
    const dropdownBtn = productRow.querySelector(`[onclick*="'${field}-dropdown-${rowIndex}'"]`);
    const originalContent = dropdownBtn.innerHTML;
    dropdownBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i> Updating...';

    // Send AJAX request
    fetch(`/admin/products/${productId}`, {
        method: 'PUT',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
            'X-Requested-With': 'XMLHttpRequest'
        },
        body: JSON.stringify({ [field]: value })
    })
    .then(response => {
        if (!response.ok) {
            return response.json().then(err => { throw new Error(err.message || 'Update failed'); });
        }
        return response.json();
    })
    .then(data => {
        if (data.success) {
            // Update the UI text
            const spanElement = dropdownBtn.querySelector('span');
            if (spanElement) {
                spanElement.textContent = value;
            }
            showAlert('Product updated successfully!', 'success');
        } else {
            throw new Error(data.message || 'Update failed');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showAlert('Error updating product: ' + error.message, 'error');
    })
    .finally(() => {
        // Restore original button content and close dropdown
        dropdownBtn.innerHTML = originalContent;
        dropdownBtn.querySelector('span').textContent = value; // Ensure text is updated even on failure to restore
        const dropdownContent = document.getElementById(`${field}-dropdown-${rowIndex}`);
        if (dropdownContent) {
            dropdownContent.classList.remove('show');
        }
    });
}

// Show a toast-like alert message
function showAlert(message, type = 'info') {
    const alertDiv = document.createElement('div');
    alertDiv.className = `fixed top-5 right-5 z-50 px-4 py-3 rounded-lg shadow-lg text-white ${
        type === 'success' ? 'bg-green-500' :
        type === 'error' ? 'bg-red-500' :
        'bg-blue-500'
    }`;
    alertDiv.innerHTML = `<span>${message}</span>`;
    document.body.appendChild(alertDiv);
    setTimeout(() => alertDiv.remove(), 3000);
}

// Close dropdowns when clicking outside of them
document.addEventListener('click', function(event) {
    if (!event.target.closest('.dropdown')) {
        document.querySelectorAll('.dropdown-content.show').forEach(dropdown => {
            dropdown.classList.remove('show');
        });
    }
});

// Add data-row-index to each table row for easier selection in JS
document.addEventListener('DOMContentLoaded', function() {
    const rows = document.querySelectorAll('tbody tr');
    rows.forEach((row, index) => {
        // The loop->iteration is 1-based, so we use it directly
        row.setAttribute('data-row-index', index + 1);
    });
});
