@extends('layouts.app')

@section('title', 'Products List')

@section('content')
<div class="container mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <div class="bg-white rounded-lg shadow-md overflow-hidden">
        <div class="bg-gradient-to-r from-blue-600 to-purple-700 px-6 py-4">
            <h1 class="text-2xl font-bold text-white flex items-center"><i class="fas fa-boxes-stacked mr-3"></i>Product Information</h1>
            <p class="text-blue-100 mt-1">Click on dropdowns to change values</p>
        </div>

        <!-- Session Status -->
        <div class="px-6 py-4">
            @if (session('success'))
                <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4" role="alert">
                    <p class="font-bold">Success</p>
                    <p>{{ session('success') }}</p>
                </div>
            @endif
        </div>

        <!-- Statistics Cards -->
        @if(isset($stats))
        <div class="px-6 py-4 bg-gray-50 border-b border-gray-200">
            <div class="grid grid-cols-2 sm:grid-cols-3 lg:grid-cols-6 gap-4">
                <div class="bg-white rounded-lg shadow p-4 text-center">
                    <p class="text-sm font-medium text-gray-500">Total Products</p>
                    <p class="text-2xl font-bold text-gray-800">{{ $stats['total_products'] ?? 'N/A' }}</p>
                </div>
                <div class="bg-white rounded-lg shadow p-4 text-center">
                    <p class="text-sm font-medium text-gray-500">In Warehouse</p>
                    <p class="text-2xl font-bold text-indigo-600">{{ $stats['in_warehouse'] ?? 'N/A' }}</p>
                </div>
                <div class="bg-white rounded-lg shadow p-4 text-center">
                    <p class="text-sm font-medium text-gray-500">Shipped</p>
                    <p class="text-2xl font-bold text-yellow-600">{{ $stats['shipped'] ?? 'N/A' }}</p>
                </div>
                <div class="bg-white rounded-lg shadow p-4 text-center">
                    <p class="text-sm font-medium text-gray-500">Delivered</p>
                    <p class="text-2xl font-bold text-green-600">{{ $stats['delivered'] ?? 'N/A' }}</p>
                </div>
                <div class="bg-white rounded-lg shadow p-4 text-center">
                    <p class="text-sm font-medium text-gray-500">Sold</p>
                    <p class="text-2xl font-bold text-purple-600">{{ $stats['sold'] ?? 'N/A' }}</p>
                </div>
                <div class="bg-white rounded-lg shadow p-4 text-center">
                    <p class="text-sm font-medium text-gray-500">Returned</p>
                    <p class="text-2xl font-bold text-red-600">{{ $stats['returned'] ?? 'N/A' }}</p>
                </div>
            </div>
        </div>
        @endif


        <!-- Search and Filter Section -->
        <div class="px-6 py-4 bg-gray-50 border-b border-gray-200">
            <form method="GET" action="{{ route('admin.products') }}" class="flex flex-col md:flex-row gap-4">
                <div class="relative flex-grow">
                    <input type="text" name="search" value="{{ $search }}" placeholder="Search products..."
                           class="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                    <i class="fas fa-search absolute left-3 top-3 text-gray-400"></i>
                </div>
                <div class="flex flex-col sm:flex-row gap-2">
                    <select name="quality" class="bg-white px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                        <option value="">All Qualities</option>
                        <option value="Authentic" {{ $quality == 'Authentic' ? 'selected' : '' }}>Authentic</option>
                        <option value="Premium" {{ $quality == 'Premium' ? 'selected' : '' }}>Premium</option>
                        <option value="Standard" {{ $quality == 'Standard' ? 'selected' : '' }}>Standard</option>
                    </select>
                    <select name="color" class="bg-white px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                        <option value="">All Colors</option>
                        <option value="Cyan" {{ $color == 'Cyan' ? 'selected' : '' }}>Cyan</option>
                        <option value="White" {{ $color == 'White' ? 'selected' : '' }}>White</option>
                        <option value="Black" {{ $color == 'Black' ? 'selected' : '' }}>Black</option>
                    </select>
                    <button type="submit" class="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"><i class="fas fa-filter mr-2"></i>Filter</button>
                </div>
            </form>
        </div>

        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr class="text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        <th class="px-6 py-3"><i class="fas fa-hashtag mr-1"></i>#</th>
                        <th class="px-6 py-3"><i class="fas fa-barcode mr-1"></i>Serial No</th>
                        <th class="px-6 py-3"><i class="fas fa-image mr-1"></i>Image</th>
                        <th class="px-6 py-3"><i class="fas fa-box mr-1"></i>Product Name</th>
                        <th class="px-6 py-3"><i class="fas fa-heart-pulse mr-1"></i>Condition</th>
                        <th class="px-6 py-3"><i class="fas fa-truck-fast mr-1"></i>Status</th>
                        <th class="px-6 py-3"><i class="fas fa-star mr-1"></i>Quality</th>
                        <th class="px-6 py-3"><i class="fas fa-mobile-screen-button mr-1"></i>IMEI</th>
                        <th class="px-6 py-3"><i class="fas fa-palette mr-1"></i>Color</th>
                        <th class="px-6 py-3"><i class="fas fa-building mr-1"></i>Distributor</th>
                        <th class="px-6 py-3"><i class="fas fa-store mr-1"></i>Retailer</th>
                        <th class="px-6 py-3"><i class="fas fa-user mr-1"></i>Client</th>
                        <th class="px-6 py-3"><i class="fas fa-cogs mr-1"></i>Actions</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    @foreach($products as $product)
                    <tr class="hover:bg-gray-50 transition-shadow duration-200 hover:shadow-md" data-product-id="{{ $product->id }}">
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">{{ $loop->iteration }}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500 font-mono">{{ $product->serial_no }}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            <img src="{{ $product->image ? asset('storage/' . $product->image) : 'https://via.placeholder.com/150' }}"
                                 alt="{{ $product->product_name }}"
                                 class="h-12 w-12 object-cover rounded-md cursor-pointer hover:opacity-75"
                                 onclick="openImageEditModal('{{ $product->id }}', '{{ $product->image ? asset('storage/' . $product->image) : 'https://via.placeholder.com/150' }}')">
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            <div class="dropdown">
                                <div id="product_name-btn-{{ $loop->iteration }}" class="dropdown-btn" onclick="toggleDropdown('product_name-dropdown-{{ $loop->iteration }}')">
                                    <span>{{ $product->product_name }}</span>
                                    <i class="fas fa-chevron-down text-xs ml-1"></i>
                                </div>
                                <div id="product_name-dropdown-{{ $loop->iteration }}" class="dropdown-content">
                                    {{-- This should ideally come from your controller --}}
                                    @php $productNames = ['Nokia 105 (2023) +', 'Nokia 105 (2022) +', 'Nokia 210', 'Nokia 5310']; @endphp
                                    @foreach($productNames as $name)
                                        {{-- Use addslashes to escape quotes in product names --}}
                                        <a onclick="updateProductField({{ $loop->parent->iteration }}, 'product_name', '{{ addslashes($name) }}')">{{ $name }}</a>
                                    @endforeach
                                </div>
                            </div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">{{ $product->condition }}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            <div class="dropdown">
                                <div class="dropdown-btn" onclick="toggleDropdown('status-dropdown-{{ $loop->iteration }}')">
                                    <span id="status-badge-{{ $loop->iteration }}" class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full @switch($product->status) @case('In Warehouse') bg-blue-100 text-blue-800 @break @case('Shipped') bg-yellow-100 text-yellow-800 @break @case('Delivered') bg-green-100 text-green-800 @break @case('Sold') bg-purple-100 text-purple-800 @break @case('Returned') bg-red-100 text-red-800 @break @default bg-gray-100 text-gray-800 @endswitch">
                                        {{ $product->status }}
                                    </span>
                                    <i class="fas fa-chevron-down text-xs ml-1"></i>
                                </div>
                                <div id="status-dropdown-{{ $loop->iteration }}" class="dropdown-content">
                                    <a onclick="updateProductField({{ $loop->iteration }}, 'status', 'In Warehouse')">In Warehouse</a>
                                    <a onclick="updateProductField({{ $loop->iteration }}, 'status', 'Shipped')">Shipped</a>
                                    <a onclick="updateProductField({{ $loop->iteration }}, 'status', 'Delivered')">Delivered</a>
                                    <a onclick="updateProductField({{ $loop->iteration }}, 'status', 'Sold')">Sold</a>
                                    <a onclick="updateProductField({{ $loop->iteration }}, 'status', 'Returned')">Returned</a>
                                </div>
                            </div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            <div class="dropdown">
                                <div id="quality-btn-{{ $loop->iteration }}" class="dropdown-btn" onclick="toggleDropdown('quality-dropdown-{{ $loop->iteration }}')">
                                    <span>{{ $product->quality }}</span>
                                    <i class="fas fa-chevron-down text-xs ml-1"></i>
                                </div>
                                <div id="quality-dropdown-{{ $loop->iteration }}" class="dropdown-content">
                                    <a onclick="updateProductField({{ $loop->iteration }}, 'quality', 'Authentic')">Authentic</a>
                                    <a onclick="updateProductField({{ $loop->iteration }}, 'quality', 'Premium')">Premium</a>
                                    <a onclick="updateProductField({{ $loop->iteration }}, 'quality', 'Standard')">Standard</a>
                                    <a onclick="updateProductField({{ $loop->iteration }}, 'quality', 'Economy')">Economy</a>
                                </div>
                            </div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500 font-mono">{{ $product->imei }}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            <div class="dropdown">
                                <div id="color-btn-{{ $loop->iteration }}" class="dropdown-btn" onclick="toggleDropdown('color-dropdown-{{ $loop->iteration }}')">
                                    <span>{{ $product->color }}</span>
                                    <i class="fas fa-chevron-down text-xs ml-1"></i>
                                </div>
                                <div id="color-dropdown-{{ $loop->iteration }}" class="dropdown-content">
                                    <a onclick="updateProductField({{ $loop->iteration }}, 'color', 'Cyan')">Cyan</a>
                                    <a onclick="updateProductField({{ $loop->iteration }}, 'color', 'White')">White</a>
                                    <a onclick="updateProductField({{ $loop->iteration }}, 'color', 'Black')">Black</a>
                                    <a onclick="updateProductField({{ $loop->iteration }}, 'color', 'Red')">Red</a>
                                </div>
                            </div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">{{ $product->distributor_name }}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">{{ $product->retailer_name }}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">{{ $product->client_name }}</td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                            <div class="flex items-center space-x-4">
                                <a href="{{ route('admin.products.edit', $product->id) }}" class="text-indigo-600 hover:text-indigo-900" title="Edit Product">
                                    <i class="fas fa-pencil-alt"></i>
                                </a>
                                <form action="{{ route('admin.products.delete', $product->id) }}" method="POST" onsubmit="return confirm('Are you sure you want to delete this product?');">
                                    @csrf
                                    @method('DELETE')
                                    <button type="submit" class="text-red-600 hover:text-red-900" title="Delete Product"><i class="fas fa-trash"></i></button>
                                </form>
                            </div>
                        </td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>

        <!-- Pagination -->
        <div class="bg-gray-50 px-6 py-3 flex items-center justify-between border-t border-gray-200">
            <div class="text-sm text-gray-700">
                Showing <span class="font-medium">{{ $products->firstItem() }}</span> to <span class="font-medium">{{ $products->lastItem() }}</span> of <span class="font-medium">{{ $products->total() }}</span> results
            </div>
            <div class="flex space-x-2">
                @if($products->onFirstPage())
                    <span class="px-4 py-2 text-sm bg-gray-200 text-gray-500 rounded-lg cursor-not-allowed">Previous</span>
                @else
                    <a href="{{ $products->previousPageUrl() . '&' . http_build_query(request()->except('page')) }}" class="px-4 py-2 text-sm bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">Previous</a>
                @endif

                @if($products->hasMorePages())
                    <a href="{{ $products->nextPageUrl() . '&' . http_build_query(request()->except('page')) }}" class="px-4 py-2 text-sm bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">Next</a>
                @else
                    <span class="px-4 py-2 text-sm bg-gray-200 text-gray-500 rounded-lg cursor-not-allowed">Next</span>
                @endif
            </div>
        </div>
    </div>
</div>

<!-- Image Edit Modal -->
<div id="image-edit-modal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50 hidden">
    <div class="relative top-20 mx-auto p-5 border w-full max-w-lg shadow-lg rounded-md bg-white">
        <div class="flex justify-between items-center pb-3 border-b">
            <h3 class="text-2xl font-bold text-gray-800">Edit Product Image</h3>
            <button id="close-image-modal-btn" class="text-gray-600 hover:text-gray-900">
                <i class="fas fa-times fa-lg"></i>
            </button>
        </div>
        <div class="mt-5">
            <form id="image-edit-form" enctype="multipart/form-data">
                <div id="image-modal-errors" class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 mb-4 hidden" role="alert"></div>
                <input type="hidden" id="edit_image_product_id" name="product_id">
                <input type="hidden" name="_method" value="PUT">

                <div class="flex flex-col items-center">
                    <img id="edit_image_preview" src="" alt="Current Image" class="h-48 w-48 object-cover rounded-md bg-gray-100 mb-4">
                    <input type="file" name="image" id="edit_image_input" class="block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-sm file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100" required>
                </div>

                <div class="mt-8 flex justify-end space-x-4">
                    <button type="button" id="cancel-image-modal-btn" class="bg-gray-200 text-gray-800 px-6 py-2 rounded-lg hover:bg-gray-300 transition-colors">Cancel</button>
                    <button type="submit" class="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700 transition-colors flex items-center"><i class="fas fa-save mr-2"></i>Update Image</button>
                </div>
            </form>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    const imageModal = document.getElementById('image-edit-modal');
    const closeBtn = document.getElementById('close-image-modal-btn');
    const cancelBtn = document.getElementById('cancel-image-modal-btn');
    const imageForm = document.getElementById('image-edit-form');
    const imageInput = document.getElementById('edit_image_input');
    const imagePreview = document.getElementById('edit_image_preview');

    function closeModal() {
        imageModal.classList.add('hidden');
    }

    closeBtn.addEventListener('click', closeModal);
    cancelBtn.addEventListener('click', closeModal);

    imageInput.addEventListener('change', function(e) {
        const file = e.target.files[0];
        if (file) {
            const reader = new FileReader();
            reader.onload = function(event) {
                imagePreview.src = event.target.result;
            }
            reader.readAsDataURL(file);
        }
    });

    imageForm.addEventListener('submit', function(e) {
        e.preventDefault();
        const productId = document.getElementById('edit_image_product_id').value;
        const formData = new FormData(this);

        fetch(`/admin/products/${productId}`, {
            method: 'POST', // FormData with files requires POST
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'X-Requested-With': 'XMLHttpRequest',
                'Accept': 'application/json',
            },
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Reload the page to see the new image
                location.reload();
            } else {
                const errorContainer = document.getElementById('image-modal-errors');
                let errorHtml = '<p class="font-bold">Error:</p><ul>';
                if (data.errors) {
                    for (const field in data.errors) {
                        errorHtml += `<li>${data.errors[field][0]}</li>`;
                    }
                } else {
                    errorHtml += `<li>${data.message || 'An unknown error occurred.'}</li>`;
                }
                errorHtml += '</ul>';
                errorContainer.innerHTML = errorHtml;
                errorContainer.classList.remove('hidden');
            }
        })
        .catch(console.error);
    });
});

function openImageEditModal(productId, currentImageUrl) {
    const imageModal = document.getElementById('image-edit-modal');
    document.getElementById('edit_image_product_id').value = productId;
    document.getElementById('edit_image_preview').src = currentImageUrl;
    document.getElementById('image-edit-form').reset(); // Reset file input
    document.getElementById('image-modal-errors').classList.add('hidden');
    imageModal.classList.remove('hidden');
}
</script>
@endpush
